# 📊 Struktura bazy danych Ayurwell Platform

## 🔑 Tabele główne (UŻYWAMY)

### `crm_users` - Użytkownicy systemu
```sql
CREATE TABLE users (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  email TEXT UNIQUE,
  password TEXT,
  role TEXT DEFAULT 'user',                 -- 'user', 'sales', 'admin'
  crm_access INTEGER DEFAULT 0,             -- 0 = brak, 1 = dostęp
  crm_access_granted BOOLEAN DEFAULT 0,     -- duplikat crm_access
  user_type TEXT DEFAULT 'client',          -- 'client', 'business'
  first_name TEXT,
  last_name TEXT,
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP
);
```

### `crm_leads` - Leady CRM
```sql
CREATE TABLE crm_leads (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  company_name TEXT NOT NULL,
  contact_name TEXT,
  phone TEXT,
  email TEXT,
  website TEXT,
  lead_source TEXT,
  lead_status TEXT DEFAULT 'new',
  category TEXT,
  address TEXT,
  city TEXT,
  notes TEXT,
  assigned_to INTEGER,                      -- FK: crm_users.id
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  last_contact_date DATETIME,
  FOREIGN KEY (assigned_to) REFERENCES crm_users(id)
);
```

### `crm_activities` - Aktywności CRM
```sql
CREATE TABLE crm_activities (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  lead_id INTEGER,                          -- FK: crm_leads.id
  activity_type TEXT,
  description TEXT,
  scheduled_date DATETIME,
  completed_date DATETIME,
  completed_by INTEGER,                     -- FK: crm_users.id
  notes TEXT,
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (lead_id) REFERENCES crm_leads(id),
  FOREIGN KEY (completed_by) REFERENCES crm_users(id)
);
```

### `crm_sales_team` - Zespół sprzedaży
```sql
CREATE TABLE crm_sales_team (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  user_id INTEGER UNIQUE,                   -- FK: crm_users.id
  position TEXT,
  hire_date DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES crm_users(id)
);
```

### `platform_clients` - Rozszerzone profile
```sql
CREATE TABLE platform_clients (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  user_id INTEGER UNIQUE,                   -- FK: crm_users.id
  first_name TEXT,
  last_name TEXT,
  phone TEXT,
  date_of_birth DATE,
  gender TEXT,
  address TEXT,
  city TEXT,
  postal_code TEXT,
  country TEXT DEFAULT 'Poland',
  avatar_url TEXT,
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES crm_users(id)
);
```

### `user_settings` - Ustawienia użytkowników
```sql
CREATE TABLE user_settings (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  user_id INTEGER,                          -- FK: crm_users.id
  settings_json TEXT,                       -- JSON string
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES crm_users(id)
);
```

## ⚠️ Tabele duplikaty (DO USUNIĘCIA)
- `leads` → użyj `crm_leads`
- `lead_activities` → użyj `crm_activities`  
- `sales_team` → użyj `crm_sales_team`
- `users` → użyj `crm_users`

## 🔐 System uprawnień

### Rejestracja (domyślnie):
```javascript
role: 'user'
crm_access: 0
user_type: 'client'
```

### Ręczne nadanie uprawnień CRM:
```sql
UPDATE crm_users SET 
  crm_access = 1, 
  role = 'sales' 
WHERE email = 'sprzedawca@firma.pl';
```

### Role i uprawnienia:
- `user` + `crm_access: 0` → Panel klienta tylko
- `sales` + `crm_access: 1` → Dostęp do CRM
- `admin` + `crm_access: 1` → Pełny dostęp

## 🌍 Kompatybilność lokalnie/serwer

### Lokalne (development):
```javascript
// config/config.js
DB_PATH: './database.sqlite'
NODE_ENV: 'development'
```

### Serwer (production):
```javascript
// Railway PostgreSQL
DATABASE_URL: process.env.DATABASE_URL
NODE_ENV: 'production'
```

### Adapter bazy danych:
```javascript
// models/database.js
const db = process.env.NODE_ENV === 'production' 
  ? initPostgreSQL() 
  : initSQLite();
```

## 📝 Migracja danych (przyszłość)
Gdy przejdziemy na PostgreSQL:
1. Export SQLite → SQL dump
2. Konwersja typów (INTEGER → SERIAL, DATETIME → TIMESTAMP)
3. Import do PostgreSQL
4. Update connection string

## 🎯 Status gotowości
✅ Struktura kompatybilna z kodem backend  
✅ JWT system ready (role, crm_access)  
✅ Wszystkie tabele CRM obecne  
⚠️ Wymagane czyszczenie duplikatów